---
title: Create custom headers for Cloudflare Access-protected origins with
  Workers · Cloudflare Zero Trust docs
description: This tutorial covers how to use a Cloudflare Worker to add custom
  headers to traffic. The headers will be sent to origin services protected by
  Cloudflare Access.
lastUpdated: 2025-10-13T13:40:40.000Z
chatbotDeprioritize: false
tags: JavaScript
source_url:
  html: https://developers.cloudflare.com/cloudflare-one/tutorials/access-workers/
  md: https://developers.cloudflare.com/cloudflare-one/tutorials/access-workers/index.md
---

This tutorial covers how to use a [Cloudflare Worker](https://developers.cloudflare.com/workers/) to add custom HTTP headers to traffic, and how to send those custom headers to your origin services protected by [Cloudflare Access](https://developers.cloudflare.com/cloudflare-one/policies/access/).

Some applications and networking implementations require specific custom headers to be passed to the origin, which can be difficult to implement for traffic moving through a Zero Trust proxy. You can configure a Worker to send the [user authorization headers](https://developers.cloudflare.com/cloudflare-one/identity/authorization-cookie/) required by Access.

***

## Before you begin

* Secure your origin server with Cloudflare Access

## Before you begin

1. In the [Cloudflare dashboard](https://dash.cloudflare.com/), go to the **Workers & Pages** page.

   [Go to **Workers & Pages**](https://dash.cloudflare.com/?to=/:account/workers-and-pages)

2. If this is your first Worker, select **Create Worker**. Otherwise, select **Create application**, then select **Create Worker**.

3. Enter an identifiable name for the Worker, then select **Deploy**.

4. Select **Edit code**.

5. Input the following Worker:

* JavaScript

  ```js
  export default {
    async fetch(request, env, ctx) {
      const { headers } = request;
      const cfaccessemail = headers.get("cf-access-authenticated-user-email");


      const requestWithID = new Request(request);
      requestWithID.headers.set("company-user-id", cfaccessemail);


      return fetch(requestWithID);
    },
  };
  ```

* TypeScript

  ```ts
  export default {
    async fetch(request, env, ctx): Promise<Response> {
      const { headers } = request;
      const cfaccessemail = headers.get("cf-access-authenticated-user-email");


      const requestWithID = new Request(request);
      requestWithID.headers.set("company-user-id", cfaccessemail);


      return fetch(requestWithID);
    },
  } satisfies ExportedHandler<Env>;
  ```

1. Select **Save and deploy**.

Your Worker is now ready to send custom headers to your Access-protected origin services.

## Apply the Worker to your hostname

1. Select the Worker you created, then go to **Triggers**.
2. In **Routes**, select **Add route**.
3. Enter the hostname and zone for your origin, then select **Add route**.

The Worker will now insert a custom header into requests that match the defined route. For example:

```http
"Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7",
    "Accept-Encoding": "gzip",
    "Accept-Language": "en-US,en;q=0.9",
    "Cf-Access-Authenticated-User-Email": "user@example.com",
    "Company-User-Id": "user@example.com",
    "Connection": "keep-alive"
```
