---
title: Wrangler configuration · Cloudflare Sandbox SDK docs
description: "The minimum required configuration for using Sandbox SDK:"
lastUpdated: 2025-10-17T21:25:11.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/sandbox/configuration/wrangler/
  md: https://developers.cloudflare.com/sandbox/configuration/wrangler/index.md
---

## Minimal configuration

The minimum required configuration for using Sandbox SDK:

```jsonc
{
  "name": "my-sandbox-worker",
  "main": "src/index.ts",
  "compatibility_date": "2025-10-13",
  "compatibility_flags": ["nodejs_compat"],
  "containers": [
    {
      "class_name": "Sandbox",
      "image": "./Dockerfile",
    },
  ],
  "durable_objects": {
    "bindings": [
      {
        "class_name": "Sandbox",
        "name": "Sandbox",
      },
    ],
  },
  "migrations": [
    {
      "new_sqlite_classes": ["Sandbox"],
      "tag": "v1",
    },
  ],
}
```

## Required settings

### containers

Each container is backed by its own Durable Object. The container image contains your runtime environment.

```jsonc
{
  "containers": [
    {
      "class_name": "Sandbox",
      "image": "./Dockerfile",
    },
  ],
}
```

**Parameters**:

* **class\_name** (string, required) - Must match the `class_name` of the Durable Object.
* **image** (string, required) - The Docker image to use. Must match your npm package version.

See [Dockerfile reference](https://developers.cloudflare.com/sandbox/configuration/dockerfile/) for information on customizing your Docker image.

### durable\_objects.bindings

Bind the Sandbox Durable Object to your Worker:

```jsonc
{
  "durable_objects": {
    "bindings": [
      {
        "class_name": "Sandbox",
        "name": "Sandbox",
      },
    ],
  },
}
```

**Parameters**:

* **class\_name** (string, required) - Must match the `class_name` of the container configuration.
* **name** (string, required) - The binding name you'll use in your code. Conventionally `"Sandbox"`.

### migrations

Required for Durable Object initialization:

```jsonc
{
  "migrations": [
    {
      "new_sqlite_classes": ["Sandbox"],
      "tag": "v1",
    },
  ],
}
```

This tells Cloudflare to initialize the Sandbox Durable Object with SQLite storage.

## Optional settings

These settings are illustrative and not required for basic usage.

### Environment variables

Pass configuration to your Worker:

```jsonc
{
  "vars": {
    "ENVIRONMENT": "production",
    "LOG_LEVEL": "info",
  },
}
```

Access in your Worker:

```typescript
export default {
  async fetch(request: Request, env: Env): Promise<Response> {
    console.log(`Running in ${env.ENVIRONMENT} mode`);
    // ...
  },
};
```

### Secrets

Store sensitive values securely:

```bash
# Set secrets via CLI (never commit these)
wrangler secret put ANTHROPIC_API_KEY
wrangler secret put GITHUB_TOKEN
wrangler secret put DATABASE_URL
```

Access like environment variables:

```typescript
interface Env {
  Sandbox: DurableObjectNamespace;
  ANTHROPIC_API_KEY: string;
  GITHUB_TOKEN: string;
}
```

### Cron triggers

Run sandboxes on a schedule:

```jsonc
{
  "triggers": {
    "crons": ["0 0 * * *"], // Daily at midnight
  },
}
```

```typescript
export default {
  async scheduled(event: ScheduledEvent, env: Env): Promise<void> {
    const sandbox = getSandbox(env.Sandbox, "scheduled-task");
    await sandbox.exec("python3 /workspace/daily-report.py");
    await sandbox.destroy();
  },
};
```

## Troubleshooting

### Binding not found

**Error**: `TypeError: env.Sandbox is undefined`

**Solution**: Ensure your `wrangler.jsonc` includes the Durable Objects binding:

```jsonc
{
  "durable_objects": {
    "bindings": [
      {
        "class_name": "Sandbox",
        "name": "Sandbox",
      },
    ],
  },
}
```

### Missing migrations

**Error**: Durable Object not initialized

**Solution**: Add migrations for the Sandbox class:

```jsonc
{
  "migrations": [
    {
      "new_sqlite_classes": ["Sandbox"],
      "tag": "v1",
    },
  ],
}
```

## Related resources

* [Wrangler documentation](https://developers.cloudflare.com/workers/wrangler/) - Complete Wrangler reference
* [Durable Objects setup](https://developers.cloudflare.com/durable-objects/get-started/) - DO-specific configuration
* [Dockerfile reference](https://developers.cloudflare.com/sandbox/configuration/dockerfile/) - Custom container images
* [Environment variables](https://developers.cloudflare.com/sandbox/configuration/environment-variables/) - Passing configuration to sandboxes
* [Get Started guide](https://developers.cloudflare.com/sandbox/get-started/) - Initial setup walkthrough
